<?php

namespace App\Service;

use DateTime;
use App\Entity\Affaire;
use App\Entity\DetailAffaire;
use App\Repository\AffaireRepository;
use PhpOffice\PhpSpreadsheet\IOFactory;
use Doctrine\ORM\EntityManagerInterface;
use PhpOffice\PhpSpreadsheet\Shared\Date;


class ExcelImportService
{
    private AffaireRepository $affaireRepository;

    public function __construct(AffaireRepository $affaireRepository, private EntityManagerInterface $em)
    {
        $this->affaireRepository = $affaireRepository;
    }
    public function importExcelData(string $filePath, string $originalFilename, bool $force = false, DateTime $dateDispo, $affaire): array
    {
        $spreadsheet = IOFactory::load($filePath);
        $targetSheet = null;

        // Nom recherché (propre et standardisé)
        $searchedSheetName = 'feuille de colle';

        // Normalisation simple
        function normalizeSheetName(string $name): string {
            return strtolower(trim(preg_replace('/[^\p{L}\d]/u', '', iconv('UTF-8', 'ASCII//TRANSLIT', $name))));
        }

        // Parcours toutes les feuilles pour trouver celle qui correspond approximativement
        foreach ($spreadsheet->getAllSheets() as $sheet) {
            $currentSheetNameNormalized = normalizeSheetName($sheet->getTitle());
            $searchedSheetNameNormalized = normalizeSheetName($searchedSheetName);

            if ($currentSheetNameNormalized === $searchedSheetNameNormalized) {
                $targetSheet = $sheet;
                break;
            }
        }

        if (!$targetSheet) {
            throw new \Exception('Feuille recherchée introuvable.');
        }else{
            $worksheet = $targetSheet;
        }

        // Test si pas déjà imoprté :
        $nomChantier = $worksheet->getCell('V6')->getCalculatedValue();
        $existant = $this->affaireRepository->findByNomChantierEtFichier($nomChantier, $originalFilename);

        if ($existant && !$force && $affaire == null) {
            return [
                'warning' => true,
                'nomChantier' => $nomChantier,
                'nomFichier' => $originalFilename,
                'message' => "Le fichier \"$originalFilename\" a déjà été importé pour le chantier \"$nomChantier\". Voulez-vous continuer ?"
            ];
        }

        // SI affaire est null on crée une affaire sinon on passe directement à l'ajout de detail
        if($affaire == null){
            // Récupérer les informations de l'affaire
            $affaire = new Affaire();
            //Date de Mise à Dispo
            $affaire->setDateMiseDispo($dateDispo);            
            $semainePlanification = (int) ($dateDispo->format('W') - 6); 
            $affaire->setNumeroSemainePlanification($semainePlanification);
            // DATE FABRICATION
            $dateFabrication = $worksheet->getCell('D8')->getCalculatedValue();
            if (!empty($dateFabrication) && is_numeric($dateFabrication)) {
                try {
                    $dateFabrication = Date::excelToDateTimeObject($dateFabrication);
                } catch (\Throwable $e) {
                    // Optionnel : journalise l'erreur, ou garde la date à null
                    $dateFabrication = null;
                }
            }
            $affaire->setDateFabrication($dateFabrication);
            

            // DATE LIVRAISON
            $dateLivraison = $worksheet->getCell('D9')->getCalculatedValue();
            if (!empty($dateLivraison) && is_numeric($dateLivraison)) {
                try {
                    $dateLivraison = Date::excelToDateTimeObject($dateLivraison);
                } catch (\Throwable $e) {
                    // Optionnel : journalise l'erreur, ou garde la date à null
                    $dateLivraison = null;
                }
            }
            $affaire->setDateLivraison($dateLivraison);
            // DESSINATEUR
            $dessinateur = $worksheet->getCell('A10')->getCalculatedValue();
            $affaire->setDessinateur($dessinateur);
            // ASSEMBLAGE ATELIER
            $assemblageAtelier = $worksheet->getCell('I2')->getCalculatedValue();
            if ($assemblageAtelier == "x"){ $assemblageAtelier=true;}else{ $assemblageAtelier = false;}
            $affaire->setAssemblageAtelier($assemblageAtelier);
            // TRAITEMENT
            if (strtolower($worksheet->getCell('I6')->getCalculatedValue()) == "x"){
                $traitement ="Arvonyl D18.8-L18";
            }elseif (strtolower($worksheet->getCell('I7')->getCalculatedValue()) == "x"){
                $traitement = "EXO 1000 chêne Doré";
            }elseif (strtolower($worksheet->getCell('I8')->getCalculatedValue()) == "x"){
                $traitement ="EXO 1000 incolore";
            }elseif (strtolower($worksheet->getCell('I9')->getCalculatedValue()) == "x"){
                $traitement = "SARPECO 9+";
            }else{
                $traitement ="";
            }
            $affaire->setTraitement($traitement);
            // TAILLAGE
            $taillageTete = $worksheet->getCell('N4')->getCalculatedValue();
            if (strtolower($taillageTete) == "x"){ $taillageTete=true;}else{ $taillageTete = false;}
            $affaire->setTaillageTete($taillageTete);

            $taillageRefilage = $worksheet->getCell('N5')->getCalculatedValue();
            if (strtolower($taillageRefilage) == "x"){ $taillageRefilage=true;}else{ $taillageRefilage = false;}
            $affaire->setTaillageRefilage($taillageRefilage);
            
            $taillageDelardement = $worksheet->getCell('N6')->getCalculatedValue();
            if (strtolower($taillageDelardement) == "x"){ $taillageDelardement=true;}else{ $taillageDelardement = false;}
            $affaire->setTaillageDelardement($taillageDelardement);
            
            // ESSENCE
            $essence = $worksheet->getCell('K8')->getCalculatedValue();
            $affaire->setEssence($essence);
            // COLLE
            $colle = $worksheet->getCell('K9')->getCalculatedValue();
            $affaire->setColle($colle);
            // HUMIDITE
            $humidite = $worksheet->getCell('K10')->getCalculatedValue();
            $humiditeNumericValue = null;

            if (preg_match('/([\d.,]+)%/', $humidite, $matches)) {
                // Remplace la virgule éventuelle par un point pour le décimal
                $number = str_replace(',', '.', $matches[1]);

                // Conversion en float puis division par 100 pour obtenir le format 0.12
                $humiditeNumericValue = floatval($number) / 100;
            }
            $affaire->setHumidite($humiditeNumericValue);
            
            // BATIMENT
            $batimentIndus = $worksheet->getCell('R4')->getCalculatedValue();
            if (strtolower($batimentIndus) == "x"){ $batimentIndus=true;}else{ $batimentIndus = false;}
            $affaire->setBatimentIndustriel($batimentIndus);

            $batimentAgricole = $worksheet->getCell('R5')->getCalculatedValue();
            if (strtolower($batimentAgricole) == "x"){ $batimentAgricole=true;}else{ $batimentAgricole = false;}
            $affaire->setBatimentAgricole($batimentAgricole);

            $batimentSportif = $worksheet->getCell('R6')->getCalculatedValue();
            if (strtolower($batimentSportif) == "x"){ $batimentSportif=true;}else{ $batimentSportif = false;}
            $affaire->setBatimentSportif($batimentSportif);

            $batimentEquestre = $worksheet->getCell('R7')->getCalculatedValue();
            if (strtolower($batimentEquestre) == "x"){ $batimentEquestre=true;}else{ $batimentEquestre = false;}
            $affaire->setBatimentEquestre($batimentEquestre);

            $batimentCulturel = $worksheet->getCell('R8')->getCalculatedValue();
            if (strtolower($batimentCulturel) == "x"){ $batimentCulturel=true;}else{ $batimentCulturel = false;}
            $affaire->setBatimentCulturel($batimentCulturel);

            $batimentAutre = $worksheet->getCell('R9')->getCalculatedValue();
            if (strtolower($batimentAutre) == "x"){ $batimentAutre=true;}else{ $batimentAutre = false;}
            $affaire->setBatimentAutre($batimentAutre);

            // CODE CLIENT
            $codeClient = $worksheet->getCell('X2')->getCalculatedValue();
            $affaire->setClient($codeClient);
            // REFERENCE
            $ref = $worksheet->getCell('W4')->getCalculatedValue();
            if ($ref != null){$affaire->setReference($ref);}
            // NOM CHANTIER
            $nomChantier = $worksheet->getCell('V6')->getCalculatedValue();
            $affaire->setNomChantier($nomChantier);
            // ADRESSE CHANTIER
            $adresseChantier = $worksheet->getCell('V7')->getCalculatedValue();
            $affaire->setAdresseChantier($adresseChantier);
            // FICHIER ET DATE IMPORT
            $affaire->setDateImport(new DateTime());
            $affaire->setNomFichiersExcel($originalFilename);
                
        }else{
            $affaire->setDateUpdate(new DateTime());
        }
        
        $affaire->setDateMiseDispo($dateDispo);
        $this->em->persist($affaire);
        // Récupérer toutes les lignes à partir de la ligne 14
        $rows = [];
        $highestRow = $worksheet->getHighestRow();
        $highestColumn = $worksheet->getHighestColumn();
        $importedCount = 0;

        for ($row = 14; $row <= $highestRow; $row++) {
            //$rowData = $worksheet->rangeToArray("A{$row}:{$highestColumn}{$row}", null, true, false)[0];
            $rowData = $worksheet->rangeToArray("A{$row}:{$highestColumn}{$row}", null, true, true)[0];
            $cellCoord = "A{$row}";
            $fill = $worksheet->getStyle($cellCoord)->getFill();

            $couleurFond = null;
            if ($fill->getFillType() !== \PhpOffice\PhpSpreadsheet\Style\Fill::FILL_NONE) {
                $couleurFond = $fill->getStartColor()->getRGB(); // Ex : FFFF00
            }

            if (array_filter($rowData)) {
                if($rowData[0] != null && is_numeric($rowData[0])){                    
                    
                    //$nbPiece = (int)$rowData[0];
                    //for ($i=0; $i<$nbPiece; $i++){

                        $detailAffaire = new DetailAffaire();
                        $detailAffaire->setNbPiece(1);
                        $detailAffaire->setBkgColor($couleurFond);
                        $detailAffaire->setAffaire($affaire);
                        // Conversion explicite en integer
                        $detailAffaire->setNbPiece((int)$rowData[0]);
                        $detailAffaire->setUsinage($rowData[1] ?? null);

                        $detailAffaire->setDesignation($rowData[2] ?? null);
                        $detailAffaire->setRef((int)$rowData[3] ?? null);
                        $detailAffaire->setCtf( is_numeric($rowData[4]) ? (float)str_replace(',', '.', $rowData[4]) : null);                                
                                
                        $detailAffaire->setSectionHauteur((int)$rowData[5] ?? null);
                        $detailAffaire->setSectionLargeur((int)$rowData[6] ?? null);
                        $detailAffaire->setLongueurArrondie((int)$rowData[7] ?? null);

                        // Conversion booléen
                        $detailAffaire->setDroit(strtolower($rowData[8] ?? '') === "x");
                        $detailAffaire->setLamelisation(strtolower($rowData[9] ?? '') === "x");
                                
                        $detailAffaire->setLongueurProduction((int)$rowData[10] ?? null);
                        $detailAffaire->setSectionLameLargeur((int)$rowData[11] ?? null);
                        $detailAffaire->setSectionLameHauteur((int)$rowData[12] ?? null);
                        
                        $detailAffaire->setEpaisseurRabotee((int)$rowData[13] ?? null);
                        $detailAffaire->setNbLame((int)$rowData[14] ?? null);
                        $detailAffaire->setLongueurLameAboutee((int)$rowData[15] ?? null);
                        $detailAffaire->setVolumefini( is_numeric($rowData[16]) ? (float)str_replace(',', '.', $rowData[16]) : null);
                        $detailAffaire->setRemarque($rowData[17] ?? null);

                        // Conversion booléen
                        $detailAffaire->setTac(strtolower($rowData[18] ?? '') === "x");
                        $detailAffaire->setGl28h(strtolower($rowData[19] ?? '') === "x");
                        $detailAffaire->setGl24h(strtolower($rowData[20] ?? '') === "x");
                        $detailAffaire->setGlxxhsRefile(strtolower($rowData[21] ?? '') === "x");
                        $detailAffaire->setControleDimensionnel(strtolower($rowData[22] ?? '') === "x");

                        $detailAffaire->setLongueurLameAboutee((int)$rowData[23] ?? null);
                        $detailAffaire->setVolumeBoisAbouteNonRabote( is_numeric($rowData[24]) ? (float)str_replace(',', '.', $rowData[24]) : null);
                        $detailAffaire->setSerie((int)$rowData[25] ?? null);
                        
                        // Statut : 
                        $detailAffaire->setStatut(1);
                        // Date Create
                        $detailAffaire->setDateCreate(new DateTime());
                        $this->em->persist($detailAffaire);
                    //}

                    $importedCount++;
                }
            }
        }
        $this->em->flush();

        return [            
            'rows' => $rows,
        ];
    }
}
