<?php

namespace App\Repository;

use App\Entity\Affaire;
use App\Entity\DetailAffaire;
use Doctrine\Persistence\ManagerRegistry;
use Doctrine\Bundle\DoctrineBundle\Repository\ServiceEntityRepository;

/**
 * @extends ServiceEntityRepository<DetailAffaire>
 */
class DetailAffaireRepository extends ServiceEntityRepository
{
    public function __construct(ManagerRegistry $registry)
    {
        parent::__construct($registry, DetailAffaire::class);
    }
    /**
     * @return [type]
     */
    public function getListeDetailParAffaire(Affaire $affaire) {

        $query = $this->createQueryBuilder('da')
            ->where('da.dateDelete is NULL')
            ->leftJoin('da.regroupements', 'r')
            ->andWhere('da.affaire = :affaire')
            ->andWhere('r.id IS NULL')
            ->setParameter('affaire', $affaire)
            ->orderBy('da.ref', 'ASC')
        ;
        
        return $query->getQuery()->getResult();
    }

    public function getListeDetailParAffaireRegroupee(Affaire $affaire){
        $meta = $this->getEntityManager()->getClassMetadata(\App\Entity\DetailAffaire::class);
        $fields = $meta->getFieldNames();

        $excludedFields = ['id', 'nbPiece'];
        $groupFields = array_filter($fields, fn($f) => !in_array($f, $excludedFields));

        $selectParts = array_map(fn($f) => 'da.' . $f, $groupFields);
        $select = implode(', ', $selectParts);
        $groupBy = implode(', ', $selectParts);

        $query = $this->createQueryBuilder('da')
            ->select($select . ', SUM(da.nbPiece) AS nbPieceTotal')
            ->where('da.dateDelete IS NULL')
            ->leftJoin('da.regroupements', 'r')
            ->andWhere('da.affaire = :affaire')
            ->andWhere('r.id IS NULL')
            ->groupBy($groupBy)
            ->orderBy('da.ref', 'ASC')
            ->setParameter('affaire', $affaire);

        return $query->getQuery()->getResult();

    }

    public function getListeDetailPlanification() {

        $query = $this->createQueryBuilder('da')
            ->where('da.dateDelete is NULL')            
            ->orderBy('da.id', 'ASC')
        ;
        
        return $query->getQuery()->getResult();
    }

    public function countRegroupeesParAffaires(): array
    {
        $qb = $this->createQueryBuilder('d')
            ->select('a.id AS affaire_id')
            ->addSelect('COUNT(d.id) as total')
            ->addSelect('SUM(CASE WHEN r.id IS NOT NULL THEN 1 ELSE 0 END) as regroupees')
            ->leftJoin('d.regroupements', 'r')
            ->join('d.affaire', 'a')
            ->groupBy('a.id');

        $results = $qb->getQuery()->getResult();

        $map = [];
        foreach ($results as $r) {
            $map[$r['affaire_id']] = [
                'total' => $r['total'],
                'regroupees' => $r['regroupees'],
            ];
        }

        return $map;
    }

    /**
     * Liste les poutres qui ne sont pas encore placées sur un banc et non regroupées
     * avec infos : id affaire, code poutre, etc.
     */
    public function findPoutresNonPlacees(): array
    {
        $em= $this->getEntityManager();   

        // Sous-requête : les poutres déjà placées (PoutrePosition)
        $subPlaced = $em->createQueryBuilder()
            ->select('pp.poutreId')
            ->from('App\Entity\PoutrePosition', 'pp')            
            ->getDQL();

        // Sous-requête : les poutres utilisées dans des regroupements
        $subGrouped = $em->createQueryBuilder()
            ->select('rgd.id')
            ->from('App\Entity\Regroupement', 'r')
            ->join('r.poutres', 'rgd')
            ->getDQL();

        // Requête principale
        return $this->createQueryBuilder('d')
            ->innerJoin('d.affaire', 'a')
            ->where("d.id NOT IN ($subPlaced)")
            ->andWhere("d.id NOT IN ($subGrouped)")
            ->andWhere('d.dateDelete IS NULL')
            ->andWhere('a.dateDelete IS NULL')
            ->orderBy('d.id', 'ASC')
            ->getQuery()
            ->getResult();    
    }

    public function findPoutresNonPlaceesAvecQuantiteRestante_old(): array
    {
        $em = $this->getEntityManager();

        // Requête avec LEFT JOIN sur les placements
        $qb = $em->createQueryBuilder();

        $qb->select('d', 'COUNT(pp.id) AS nbPlacees', '(d.nbPiece - COUNT(pp.id)) AS quantiteRestante')
            ->from('App\Entity\DetailAffaire', 'd')
            ->leftJoin('App\Entity\PoutrePosition', 'pp', 'WITH', 'pp.poutreId = d.id')
            ->innerJoin('d.affaire', 'a')
            ->where('d.dateDelete IS NULL')
            ->andWhere('a.dateDelete IS NULL')
            ->groupBy('d.id')
            ->having('quantiteRestante > 0')  // ne garde que celles qu'on peut encore coller
            ->orderBy('d.id', 'ASC');

        $results = $qb->getQuery()->getResult();

        // On convertit les résultats (Doctrine retourne des tableaux mixtes)
        return array_map(function ($row) {
            $detail = $row[0];
            $detail->quantiteRestante = (int) $row['quantiteRestante'];
            return $detail;
        }, $results);
    }
    
    public function findPoutresNonPlaceesAvecQuantiteRestante(): array
    {
        $em = $this->getEntityManager();
        $qb = $em->createQueryBuilder();

        $qb->select('d', 'd.nbPiece AS nbPiece', 'COUNT(pp.id) AS nbPlacees', '(d.nbPiece - COUNT(pp.id)) AS quantiteRestante')
            ->from('App\Entity\DetailAffaire', 'd')
            ->leftJoin('App\Entity\PoutrePosition', 'pp', 'WITH', 'pp.poutreId = d.id')
            ->innerJoin('d.affaire', 'a')
            ->where('d.dateDelete IS NULL')
            ->andWhere('a.dateDelete IS NULL')
            ->groupBy('d.id')
            ->having('quantiteRestante > 0')
            ->orderBy('d.id', 'ASC');

        $results = $qb->getQuery()->getResult();

        return array_map(function ($row) {
            /** @var DetailAffaire $detail */
            $detail = $row[0];
            $detail->setQuantiteRestante((int) $row['quantiteRestante']);
            return $detail;
        }, $results);

    }



    //    /**
    //     * @return DetailAffaire[] Returns an array of DetailAffaire objects
    //     */
    //    public function findByExampleField($value): array
    //    {
    //        return $this->createQueryBuilder('d')
    //            ->andWhere('d.exampleField = :val')
    //            ->setParameter('val', $value)
    //            ->orderBy('d.id', 'ASC')
    //            ->setMaxResults(10)
    //            ->getQuery()
    //            ->getResult()
    //        ;
    //    }

    //    public function findOneBySomeField($value): ?DetailAffaire
    //    {
    //        return $this->createQueryBuilder('d')
    //            ->andWhere('d.exampleField = :val')
    //            ->setParameter('val', $value)
    //            ->getQuery()
    //            ->getOneOrNullResult()
    //        ;
    //    }
}
