<?php

namespace App\Controller;

use App\Entity\Banc;
use App\Entity\PlanCollage;
use App\Form\PlanSearchType;
use App\Repository\BancRepository;
use App\Entity\ParametrageBancJour;
use Doctrine\ORM\EntityManagerInterface;
use App\Repository\PlanCollageRepository;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Routing\Attribute\Route;
use App\Repository\ParametrageBancJourRepository;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;

final class PlanificationController extends AbstractController
{
    private $em;
    public function __construct(EntityManagerInterface $em)
    {
        $this->em = $em;
    }

    #[Route('/planification/detail/{banc}/{jour}/{mois}/{annee}/{heure}/{minute}', name: 'app_planification_detail')]
    public function detailPlanification(Banc $banc, int $jour, int $mois, int $annee, string $heure = '08', string $minute = '00'): Response
    {
        $dateCollage = \DateTimeImmutable::createFromFormat('Y-m-d', sprintf('%04d-%02d-%02d', $annee, $mois, $jour));
        $heureString = sprintf('%02d:%02d', $heure, $minute);
        $heureCollage = \DateTimeImmutable::createFromFormat('H:i', $heureString);

        if (!$heureCollage) {
            $heureCollage = new \DateTimeImmutable('08:00');
        }
        
        $pbjRepo = $this->em->getRepository(ParametrageBancJour::class);
        $planCollageRepo = $this->em->getRepository(PlanCollage::class);
        $pbj = $pbjRepo->findOneBy(['banc' => $banc, 'date' => $dateCollage]);

        $datesDisponibles = $pbjRepo->findDatesValideesPourBanc($banc->getId());
        $bancsPourJour = $pbjRepo->findBancsValidesPourDate($dateCollage->format('Y-m-d'));
        $horaires = $pbjRepo->findHorairesValidesPourBancEtDate($banc->getId(), $dateCollage);
        $plansExistants = [];
        foreach ($horaires as $heure) {
            $plansExistants[$heure] = $planCollageRepo->collageExiste($banc->getId(), $dateCollage, $heure);
        }

     
        return $this->render('planification/detail.planification.html.twig', [
            'controller_name' => 'PlanificationController',
            'banc'=> $banc,
            'dateCollage' => $dateCollage,
            'heureCollage' => $heureCollage->format('H:i'),
            'pbj' => $pbj,
            'datesDisponibles' => $datesDisponibles,
            'bancsPourJour' => $bancsPourJour,
            'horaires' => $horaires,
            'plansExistants' => $plansExistants,
            'jour' => $jour,
            'mois' => $mois,
            'annee' => $annee,
            'heure' => $heure,
            'minute' => $minute,
            'PLANIF' => [
                'bancId' => $banc->getId(),
                'date' => $dateCollage->format('Y-m-d'),
                'heure' => $heureCollage->format('H:i'),
            ]
        ]);
    }

    #[Route('/planification', name: 'app_planification')]
    public function index(
        Request $request,
        EntityManagerInterface $em,
        PlanCollageRepository $planCollageRepository
    ): Response {
        // Création du formulaire de recherche (date + banc)
        $form = $this->createForm(PlanSearchType::class);
        $form->handleRequest($request);

        $plan = null;

        if ($form->isSubmitted() && $form->isValid()) {
            $data = $form->getData();
            $dateCollage = $data['dateCollage'];
            $heureCollage = $data['heureCollage'];
            $valide = $data['valide'] ?? false;
            
            $banc = $data['banc'];
            $dateCollage = \DateTimeImmutable::createFromMutable($dateCollage);
            $heureCollage = \DateTimeImmutable::createFromMutable($heureCollage);

            // Recherche d'un plan existant
            $plan = $planCollageRepository->findOneBy([
                'dateCollage' => $dateCollage,
                'banc' => $banc,
                'heureCollage' => $heureCollage,
            ]);

            // Si aucun plan, on en crée un nouveau en mémoire
            if (!$plan) {
                $plan = new PlanCollage();
                $plan->setDateCollage($dateCollage);
                $plan->setBanc($banc);
                $plan->setHeureCollage($heureCollage);
                $plan->setValide($valide);
                // ici, tu peux ajouter des poutres vides ou une structure de base si nécessaire
            }

            return $this->redirectToRoute('app_planification_detail', [
                'banc' => $banc->getId(),
                'jour' => $dateCollage->format('d'),
                'mois' => $dateCollage->format('m'),
                'annee' => $dateCollage->format('Y'),
                'heure' => $heureCollage->format('H'),
                'minute' => $heureCollage->format('i'),
            ]);
        }

        return $this->render('planification/index.html.twig', [
            'form' => $form->createView(),
            'plan' => $plan,
        ]);
    }

    #[Route('/planification/recap/{week}', name: 'planification_recap')]
    public function planificationRecapHebdo(
        int $week,
        Request $request,
        BancRepository $bancRepo,
        ParametrageBancJourRepository $pbjRepo,
        PlanCollageRepository $planCollageRepo
    ): Response {
        $annee = (int)($request->query->get('annee', date('Y')));
        $bancs = $bancRepo->findAll();
        $dates = [];

        $firstDay = new \DateTime();
        $firstDay->setISODate($annee, $week);
        for ($i = 0; $i < 7; $i++) {
            $dates[] = (clone $firstDay)->modify("+$i day");
        }

        $etat = [];
        $pcArray =[];
        foreach ($bancs as $banc) {
            foreach ($dates as $date) {
                $dateImmutable = new \DateTimeImmutable($date->format('Y-m-d'));
                $pbj = $pbjRepo->findOneBy(['banc' => $banc, 'date' => $date]);
                $etat[$banc->getId()][$date->format('Y-m-d')] = $pbj;                
                $plansDuJour = $planCollageRepo->findByBancAndDate($banc, $date);
                $pcArray[$banc->getId()][$date->format('Y-m-d')] = $plansDuJour;
            }
        }
        
        return $this->render('planification/recap.html.twig', [
            'bancs' => $bancs,
            'dates' => $dates,
            'etat' => $etat,
            'week' => $week,
            'annee' => $annee,
            'planCollage' => $pcArray
        ]);
    }
}
