<?php

namespace App\Controller;

use App\Entity\Banc;
use DateTimeImmutable;
use App\Entity\NumeroSerie;
use App\Entity\PlanCollage;
use App\Entity\PoutreStock;
use App\Entity\DetailAffaire;
use App\Entity\PoutrePosition;
use App\Entity\ParametrageBancJour;
use App\Service\NumeroSerieManager;
use Doctrine\ORM\EntityManagerInterface;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Routing\Attribute\Route;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;

final class PlanCollageController extends AbstractController
{
    #[Route('api/plans-collage', methods: ['POST'])]
    public function store(Request $request, EntityManagerInterface $em): JsonResponse
    {
        $data = json_decode($request->getContent(), true);

        if (
            !$data ||
            empty($data['dateCollage']) ||
            !is_array($data['bancs']) ||
            count(array_filter($data['bancs'], fn($couche) => !empty($couche))) === 0
        ) {
            return $this->json(['error' => 'Payload incomplet'], 400);
        }


        $banc = $em->getRepository(Banc::class)->find($data['bancId']);
        $longueurBanc = $data['longueurBanc'] ?? null;
        $largeurBanc = $data['largeurBanc'] ?? null;
        $isCopie = $data['copie'] ?? false;
        $isRedirect = $data['redirectAfter'] ?? false;
        // Pour calcul des numéros de série :
        $numeroSerieEntity = $em->getRepository(NumeroSerie::class)->findOneBy([]) ?? new NumeroSerie();
        $numeroCourant = $numeroSerieEntity->getDernierNumero();
        $regroupementMap = [];

        if ($longueurBanc === null || $largeurBanc === null) {
            return new JsonResponse(['error' => 'Dimensions du banc manquantes'], 400);
        }
        $date = new \DateTimeImmutable($data['dateCollage']);
        $heure = \DateTimeImmutable::createFromFormat('H:i', $data['heureCollage']);
        $valide = $data['valide'] ?? false;

        if ($isCopie) { // Si on décide créer une copie du plan à une date ultérieure
            // 1. Vérifier s’il existe un ParametrageBancJour valide
            $pbjRepo = $em->getRepository(ParametrageBancJour::class);
            $pbj = $pbjRepo->findOneBy([
                'banc' => $banc,
                'date' => $date,
                'parametrageValide' => true,
            ]);

            if (!$pbj) {
                return new JsonResponse([
                    'error' => "Aucun paramétrage validé pour cette date",
                ], 400);
            }

            // 2. Vérifier combien de plans sont déjà enregistrés ce jour-là
            $planRepo = $em->getRepository(PlanCollage::class);
            $plansExistants = $planRepo->count([
                'banc' => $banc,
                'dateCollage' => $date,
            ]);

            // 3. Déterminer combien de collages sont autorisés (ex : 2 max)
            $nombreMax = $pbj->getNombreCollageJour() ?? 1; // ajuster selon ton modèle
            if ($plansExistants >= $nombreMax) {
                return new JsonResponse([
                    'error' => "Nombre maximum de plans atteint pour cette journée",
                ], 400);
            }
            
            // 4 Verrifier que toutes les poutres sont en quantité suffisantes sinon on refuse la copie.
            $detailRepo = $em->getRepository(DetailAffaire::class);

            foreach ($data['bancs'] as $couche => $liste) {
                foreach ($liste as $entry) {
                    if (!empty($entry['isStockAuto'])) {
                        continue; // ⛔ ignore les poutres stock
                    }

                    $poutreId = $entry['poutreId'] ?? null;

                    // ⛔ ignorer les regroupements ou IDs virtuels
                    if (!$poutreId || !ctype_digit((string)$poutreId)) {
                        continue;
                    }

                    /** @var DetailAffaire $poutre */
                    $poutre = $detailRepo->find($poutreId);

                    if (!$poutre) {
                        return $this->json(['error' => "Poutre ID $poutreId introuvable"], 400);
                    }

                    $quantiteRestante = $poutre->getQuantiteRestante() ?? $poutre->getNbPiece();

                    if ($quantiteRestante <= 0) {
                        return $this->json([
                            'error' => "Quantité insuffisante pour la poutre « " . $poutre->getDesignation() . " »"
                        ], 400);
                    }

                    // 🟡 Optionnel : tu peux ici décrémenter manuellement pour éviter de compter deux fois la même poutre
                    // $poutre->setQuantiteRestante($quantiteRestante - 1);
                }
            }

            // ✅ OK pour copier → suite de la logique de création...
            $plan = new PlanCollage();
            $plan->setBanc($banc);
            $plan->setDateCollage($date);
            $plan->setHeureCollage($heure);
            $plan->setLargeurBanc($largeurBanc);
            $plan->setLongueurBanc($longueurBanc);
            $plan->setValide($valide);
            $em->persist($plan);

        }else{
            // Modifie un plan existant : même si la date a été changée dans le datePicker on en tient pas compte
            $plan = $em->getRepository(PlanCollage::class)->findOneBy([
                'banc' => $banc,
                'dateCollage' => $date,
                'heureCollage' => $heure,
            ]);
            if (!$plan) {
                //return new JsonResponse(['error' => "Aucun plan existant à modifier."], 404);
                $plan = new PlanCollage();
                $plan->setBanc($banc);
                $plan->setDateCollage($date);
                $plan->setHeureCollage($heure);
                $plan->setLargeurBanc($largeurBanc);
                $plan->setLongueurBanc($longueurBanc);
                $plan->setValide($valide);
                $em->persist($plan);
            }else{
                $plan->setHeureCollage($heure);
                $plan->setValide($valide);
            }
            
            
            // ⚠️ Supprimer les anciennes positions liées à ce plan
            foreach ($plan->getPositions() as $pp) {
                $em->remove($pp);
            }
            foreach($plan->getPoutreStocks() as $ps){
                $em->remove($ps);
            }
            $plan->setValide($valide);                        
        }
        $em->flush();
        foreach ($data['bancs'] as $couche => $liste) {
            foreach ($liste as $entry) {               
                if (!empty($entry['isStockAuto'])) {
                    $stock = new PoutreStock();
                    $stock->setLongueur($entry['longueur']);
                    $stock->setLargeur($entry['largeur']);
                    $stock->setCouche($entry['couche']);
                    $stock->setPositionLeft($entry['left']);
                    $stock->setPositionTop($entry['top']);
                    $stock->setDateCollage($date);
                    $stock->setBancId($data['bancId']);
                    $stock->setDesignation($entry['designation'] ?? 'STOCK AUTO');
                    $stock->setPlanCollage($plan);
                    $em->persist($stock);
                    continue;
                } 
                 // ignorer les IDs fictifs (ex: GRP-xx ou UUID)
                if (!ctype_digit((string) $entry['poutreId'])) {
                    continue;
                }
                $poutre = $em->getReference(DetailAffaire::class, $entry['poutreId']);

                $pp = new PoutrePosition();
                $pp->setPlan($plan);
                $pp->setPoutre($poutre);
                $pp->setCouche($couche);                
                $pp->setLeftMm(is_numeric($entry['left']) ? $entry['left'] : 0);
                $pp->setTopMm($entry['top']);
                $pp->setLargeur($poutre->getSectionLameLargeur() * $poutre->getNbLame());
                $pp->setLongueur($poutre->getLongueurProduction());

                if (!empty($entry['regroupementId'])) {
                    $pp->setRegroupementId($entry['regroupementId']);
                }
                //Affectation des numéros de série
                if($valide){
                    if (!empty($entry['regroupementId'])) {
                        
                        if (!isset( $regroupementMap[$entry['regroupementId'] ])) {
                            $numeroCourant = $numeroSerieEntity->incrementer();
                            $regroupementMap[$entry['regroupementId'] ] = $numeroCourant;
                        }
                        $pp->setNumeroSerie($regroupementMap[$entry['regroupementId'] ]);
                    } else {
                        $numeroCourant = $numeroSerieEntity->incrementer();
                        $pp->setNumeroSerie($numeroCourant);
                    }
                    $em->persist($numeroSerieEntity);               
                }    
                $em->persist($pp); 
                
            }
        }

        $em->persist($plan);        
        
        try {            
            $em->flush();
        } catch (\Throwable $e) {
            return new JsonResponse(['error' => $e->getMessage()], 500);
        }        
        
        $date = new \DateTime($data['dateCollage'] ?? 'now');
        $weekNumber = (int) $date->format('W');
       
        
       
        $url = $this->generateUrl('planification_recap', ['week' => $weekNumber]);
       

        return new JsonResponse(['redirect' => $url, 'isRedirect' => $isRedirect]);        
    }

    #[Route('api/plans-collage', methods: ['GET'])]
    public function show(Request $request, EntityManagerInterface $em): JsonResponse
    {
        $dateCollage = $request->query->get('date');
        $heureCollage = $request->query->get('heure');
        $bancId = $request->query->get('bancId');
        $largeurBanc = $request->query->get('largeurBanc');
        $longueurBanc =$request->query->get('longueurBanc');
        
        if (!$dateCollage ) {
            return $this->json(['error' => 'Date manquante'], 400);
        }
        if (!$heureCollage) {
            return $this->json(['error' => 'Heure manquante'], 400);
        }
        if ( !$bancId) {
            return $this->json(['error' => 'Banc manquant'], 400);
        }

        // Forcer et valider la date au format Y-m-d
        $dateStr = substr($dateCollage, 0, 10);
        try {
            $dateObj = new \DateTimeImmutable($dateStr);
        } catch (\Exception $e) {
            return $this->json(['error' => 'Date invalide', 'debug' => $dateCollage], 400);
        }

        // Nettoyer et valider l’heure
        $heureClean = substr(trim($heureCollage), 0, 5); // "08:00", sans seconde ni espace
        try {
            $heureObj = new \DateTimeImmutable($heureClean);
        } catch (\Exception $e) {
            return $this->json(['error' => 'Heure invalide', 'debug' => $heureCollage], 400);
        }

        // Requête Doctrine
        $qb = $em->createQueryBuilder();
        $qb->select('p', 'pp', 'a')
            ->from(PlanCollage::class, 'p')
            ->leftJoin('p.positions', 'pp')
            ->leftJoin('pp.poutre', 'a')
            ->where('p.dateCollage = :date')
            ->andWhere('p.heureCollage = :heure')
            ->andWhere('p.banc = :bancId')
            ->setParameter('date', $dateObj)
            ->setParameter('heure', $heureObj)
            ->setParameter('bancId', $bancId);

        $plan = $qb->getQuery()->getOneOrNullResult();

        if (!$plan) {
            return $this->json([
                'message' => 'Aucun plan trouvé',
                'debug' => [
                    'bancId' => $bancId,
                    'longueurBanc' => $longueurBanc,
                    'largeurBanc' => $largeurBanc,
                    'date' => $dateObj->format('Y-m-d H:i:s'),
                    'heure' => $heureObj->format('H:i:s'),
                ]
            ], 404);
        }

        $result = [];
        $regroupements = [];

        foreach ($plan->getPositions() as $pp) {
            $p = $pp->getPoutre();
            if (!$p) continue;

            $couche = $pp->getCouche();

            $data = [
                'id' => $p->getId(),
                'designation' => $p->getDesignation(),                
                'longueur' => $pp->getLongueur(),
                'largeur' => $pp->getLargeur(),
                'affaire' => [
                    'id' => $p->getAffaire()?->getId(),
                    'nom' => $p->getAffaire()?->getNomChantier(),
                ],
                'left' => $pp->getLeftMm(),
                'top' => $pp->getTopMm(),
            ];

            if ($pp->getRegroupementId()) {
                $regroupements[$couche][$pp->getRegroupementId()][] = $data;
                continue; // ⛔️ empêche d’ajouter aussi en dehors du regroupement
            }

            $result[$couche][] = $data;
        }
        foreach ($plan->getPoutreStocks() as $pp) {            

            $couche = $pp->getCouche();
            $data = [
                'id' => $pp->getId(),
                'designation' => $pp->getDesignation(),                
                'longueur' => $pp->getLongueur(),
                'largeur' => $pp->getLargeur(),
                'affaire' => [
                    'id' => Null,
                    'nom' => Null,
                ],
                'left' => $pp->getPositionLeft(),
                'top' => $pp->getPositionTop(),
                'isStockAuto' => true,
            ];            

            $result[$couche][] = $data;
        }

        // Ensuite, reconstituer les lignes regroupées
        foreach ($regroupements as $couche => $groupes) {
            foreach ($groupes as $id => $elements) {
                $result[$couche][] = [
                    'id' => $id,
                    'designation' => 'GR-' . implode('-', array_column($elements, 'designation')),
                    'longueur' => array_sum(array_column($elements, 'longueur')),
                    'largeur' => $elements[0]['largeur'] ?? 0,
                    'affaire' => $elements[0]['affaire'] ?? [],
                    'groupes' => $elements,
                    'left' => $elements[0]['left'],
                    'top' => $elements[0]['top'],
                ];
            }
        }


        return $this->json([
            'bancId' => $bancId,
            'date' => $dateStr,
            'heure' => $heureClean,
            'bancs' => $result
        ]);
    }

    #[Route('/api/dates-parametrees', name: 'api_dates_parametrees')]
    public function datesParametrees(Request $request, EntityManagerInterface $em): JsonResponse
    {
        $bancId = $request->query->get('bancId');
        if (!$bancId) {
            return new JsonResponse(['error' => 'bancId manquant'], 400);
        }

        $repo = $em->getRepository(ParametrageBancJour::class);
        $resultats = $repo->createQueryBuilder('p')
            ->select('p.date')
            ->where('p.banc = :bancId')
            ->andWhere('p.parametrageValide = true')
            ->setParameter('bancId', $bancId)
            ->orderBy('p.date', 'ASC')
            ->getQuery()
            ->getResult();

        $dates = array_map(fn($r) => $r['date']->format('Y-m-d'), $resultats);

        return new JsonResponse($dates);
    }
    #[Route('/plan/jour/{date}/ordre', name: 'ordre_plan_jour')]
    public function trierOrdreDuJour(
        \DateTimeInterface $date,
        EntityManagerInterface $em
    ): Response {
        $positions = $em->getRepository(PoutrePosition::class)
            ->createQueryBuilder('p')
            ->join('p.plan', 'plan')
            ->where('plan.dateCollage = :date')
            ->andWhere('plan.valide = true')
            ->setParameter('date', $date->format('Y-m-d'))
            ->orderBy('p.ordreAffichage', 'ASC')
            ->getQuery()
            ->getResult();

        return $this->render('plan_collage/ordre_jour.html.twig', [
            'positions' => $positions,
            'date' => $date,
        ]);
    }

    #[Route('/plan/jour/{date}/ordre/save', name: 'ordre_plan_jour_save', methods: ['POST'])]
    public function enregistrerOrdreJour(
        \DateTimeInterface $date,
        Request $request,
        EntityManagerInterface $em
    ): Response {
        $data = json_decode($request->getContent(), true); // ['ordre' => [id1, id2, id3...]]
        $ordre = $data['ordre'] ?? [];

        $positions = $em->getRepository(PoutrePosition::class)
            ->createQueryBuilder('p')
            ->join('p.plan', 'plan')
            ->where('plan.dateCollage = :date')
            ->andWhere('plan.valide = true')
            ->andWhere('p.id IN (:ids)')
            ->setParameter('date', $date->format('Y-m-d'))
            ->setParameter('ids', $ordre)
            ->getQuery()
            ->getResult();

        // Index rapide pour assignation
        $map = [];
        foreach ($positions as $p) {
            $map[$p->getId()] = $p;
        }

        foreach ($ordre as $index => $id) {
            if (isset($map[$id])) {
                $map[$id]->setOrdreAffichage($index + 1);
            }
        }

        $em->flush();

        return new JsonResponse(['status' => 'ok']);
    }

    


}
