<?php

namespace App\Controller;

use App\Repository\BancRepository;
use App\Entity\ParametrageBancJour;
use App\Service\JoursFeriesService;
use Doctrine\ORM\EntityManagerInterface;
use App\Repository\PlanCollageRepository;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Routing\Attribute\Route;
use App\Repository\ParametrageBancJourRepository;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\Form\Extension\Core\Type\TimeType;
use Symfony\Component\Form\Extension\Core\Type\NumberType;
use Symfony\Component\Form\Extension\Core\Type\IntegerType;
use Symfony\Component\Form\Extension\Core\Type\CheckboxType;
use Symfony\Component\Form\Extension\Core\Type\TextareaType;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;

final class ParametrageBancJourController extends AbstractController
{
    #[Route('/parametrage/{date}', name: 'parametrage_jour')]
    public function parametrerJour(
        \DateTimeInterface $date,
        Request $request,
        EntityManagerInterface $em,
        BancRepository $bancRepo,
        ParametrageBancJourRepository $pbjRepo
    ): Response {
        $bancs = $bancRepo->findAll();
        $formBuilder = $this->createFormBuilder();

        foreach ($bancs as $banc) {
            $pbj = $pbjRepo->findOneBy(['banc' => $banc, 'date' => $date]);

            if (!$pbj) {
                $pbj = (new ParametrageBancJour())
                    ->setBanc($banc)
                    ->setDate($date)
                    ->setIndisponible(false)
                    ->setParametrageValide(false)
                    ->setDateCreate(new \DateTime());

                // Pré-remplir depuis les paramètres du banc
                $pbj->setLongueur($banc->getLongueurMax());
                $pbj->setLargeurMax($banc->getLargeurMax());
                $pbj->setEpaisseurMax($banc->getEpaisseurMax());
                $pbj->setNombreCouche($banc->getNombreCouche());
                //$pbj->setNombreCollageJour($banc->getNombreCollageJour());
                $pbj->setNombreCollageJour(2);
                $pbj->setEpure($banc->isEpure());
                $pbj->setTempsCollage($banc->getTempsCollage());
                $pbj->setTempsSerrage($banc->getTempsSerrage());
                //$pbj->setRemarque($banc->getRemarque());
            }

            $prefix = 'banc_' . $banc->getId() . '_';

            $formBuilder
                ->add($prefix . 'longueur', NumberType::class, ['data' => $pbj->getLongueur(), 'label' => 'Longueur', 'required' => false])
                ->add($prefix . 'largeurMax', NumberType::class, ['data' => $pbj->getLargeurMax(), 'label' => 'Largeur max', 'required' => false])
                ->add($prefix . 'epaisseurMax', NumberType::class, ['data' => $pbj->getEpaisseurMax(), 'label' => 'Épaisseur max', 'required' => false])
                ->add($prefix . 'nombreCouche', IntegerType::class, ['data' => $pbj->getNombreCouche(), 'label' => 'Nb couches', 'required' => false])
                ->add($prefix . 'nombreCollageJour', IntegerType::class, ['data' => $pbj->getNombreCollageJour(), 'label' => 'Nb collages/jour', 'required' => false])
                //->add($prefix . 'epure', CheckboxType::class, ['data' => $pbj->isEpure(), 'label' => 'Épure', 'required' => false])
                ->add($prefix . 'epure', CheckboxType::class, [
                    'data' => $pbj->isEpure(),
                    'label' => 'Épure',
                    'required' => false,
                    'row_attr' => ['class' => 'form-check form-switch mb-2'],
                    'label_attr' => ['class' => 'form-check-label'],
                    'attr' => ['class' => 'form-check-input'],
                ])
                ->add($prefix . 'tempsCollage', TimeType::class, [
                    'data' => $pbj->getTempsCollage(),
                    'label' => 'Temps de collage',
                    'widget' => 'single_text',
                    'input' => 'datetime',
                    'required' => false
                ])
                ->add($prefix . 'tempsSerrage', TimeType::class, [
                    'data' => $pbj->getTempsSerrage(),
                    'label' => 'Temps de serrage',
                    'widget' => 'single_text',
                    'input' => 'datetime',
                    'required' => false
                ])
                ->add($prefix . 'indisponible', CheckboxType::class, ['data' => $pbj->isIndisponible(), 'label' => 'Indisponible', 'required' => false])
                ->add($prefix . 'parametrageValide', CheckboxType::class, ['data' => $pbj->isParametrageValide(), 'label' => 'Paramétrage validé', 'required' => false])
                ->add($prefix . 'remarque', TextareaType::class, ['data' => $pbj->getRemarque(), 'label' => 'Remarques', 'required' => false]);
        }

        $form = $formBuilder->getForm();
        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {
            foreach ($bancs as $banc) {
                $prefix = 'banc_' . $banc->getId() . '_';

                $pbj = $pbjRepo->findOneBy(['banc' => $banc, 'date' => $date])
                    ?? (new ParametrageBancJour())
                        ->setBanc($banc)
                        ->setDate($date)
                        ->setDateCreate(new \DateTime());

                $pbj->setLongueur($form->get($prefix . 'longueur')->getData());
                $pbj->setLargeurMax($form->get($prefix . 'largeurMax')->getData());
                $pbj->setEpaisseurMax($form->get($prefix . 'epaisseurMax')->getData());
                $pbj->setNombreCouche($form->get($prefix . 'nombreCouche')->getData());
                $pbj->setNombreCollageJour($form->get($prefix . 'nombreCollageJour')->getData());
                $pbj->setEpure($form->get($prefix . 'epure')->getData());
                $pbj->setTempsCollage($form->get($prefix . 'tempsCollage')->getData());
                $pbj->setTempsSerrage($form->get($prefix . 'tempsSerrage')->getData());
                $pbj->setIndisponible($form->get($prefix . 'indisponible')->getData());
                $pbj->setParametrageValide($form->get($prefix . 'parametrageValide')->getData());
                $pbj->setRemarque($form->get($prefix . 'remarque')->getData());
                $pbj->setDateUpdate(new \DateTime());

                $em->persist($pbj);
            }

            $em->flush();
            $this->addFlash('success', 'Paramétrages enregistrés pour le ' . $date->format('d/m/Y'));
            return $this->redirectToRoute('parametrage_jour', ['date' => $date->format('Y-m-d')]);
        }

        $previousDate = (new \DateTime($date->format('Y-m-d')))->modify('-1 day')->format('Y-m-d');
        $nextDate = (new \DateTime($date->format('Y-m-d')))->modify('+1 day')->format('Y-m-d');

        return $this->render('parametrage_banc_jour/index.html.twig', [
            'form' => $form->createView(),
            'date' => $date,
            'previousDate' => $previousDate,
            'nextDate' => $nextDate,
            'bancs' => $bancs,
        ]);
    }

    
    #[Route('/parametrage/calendrier/{startDate}', name: 'parametrage_calendrier')]
    public function calendrierParametrage(
        \DateTimeInterface $startDate,
        JoursFeriesService $joursFeriesService,
        BancRepository $bancRepo,
        ParametrageBancJourRepository $pbjRepo
    ): Response {
        
        //$start = (clone $startDate)->modify('monday this week');
        $start = new \DateTime($startDate->format('Y-m-d'));
        $start->modify('monday this week');

        $dates = [];
        for ($i = 0; $i < 56; $i++) {
            $day = (clone $start)->modify("+$i day");
            $dates[] = [
                'date' => $day,
                'isWeekend' => in_array((int)$day->format('w'), [0, 6]),
                'isFerie' => $joursFeriesService->estFerie($day)
            ];
        }
        

        return $this->render('parametrage_banc_jour/calendrier.html.twig', [
            'dates' => $dates,
            'startDate' => $start,
            'prevStart' => (clone $start)->modify('-7 days')->format('Y-m-d'),
            'nextStart' => (clone $start)->modify('+7 days')->format('Y-m-d'),
        ]);
    }   

    #[Route('/parametrage/recap/{week}', name: 'parametrage_recap')]
    public function parametrageRecapHebdo(
        int $week,
        Request $request,
        BancRepository $bancRepo,
        ParametrageBancJourRepository $pbjRepo,
        PlanCollageRepository $planCollageRepo
    ): Response {
        $annee = (int)($request->query->get('annee', date('Y')));
        $bancs = $bancRepo->findAll();
        $dates = [];

        $firstDay = new \DateTime();
        $firstDay->setISODate($annee, $week);
        for ($i = 0; $i < 7; $i++) {
            $dates[] = (clone $firstDay)->modify("+$i day");
        }

        $etat = [];
        $pcArray =[];
        foreach ($bancs as $banc) {
            foreach ($dates as $date) {
                $dateImmutable = new \DateTimeImmutable($date->format('Y-m-d'));
                $pbj = $pbjRepo->findOneBy(['banc' => $banc, 'date' => $date]);
                //$pcs = $planCollageRepo->findBy(['banc' => $banc, 'dateCollage' => $dateImmutable]);                
                $etat[$banc->getId()][$date->format('Y-m-d')] = $pbj;
                //$pcArray[$banc->getId()][$date->format('Y-m-d')] = $pcs;

                $plansDuJour = $planCollageRepo->findByBancAndDate($banc, $date);
                $pcArray[$banc->getId()][$date->format('Y-m-d')] = $plansDuJour;
            }
        }
        
        return $this->render('parametrage_banc_jour/recap.html.twig', [
            'bancs' => $bancs,
            'dates' => $dates,
            'etat' => $etat,
            'week' => $week,
            'annee' => $annee,
            'planCollage' => $pcArray
        ]);
    }
    
    #[Route('/api/parametrage-banc-jour', name: 'api_parametrage_banc_jour', methods: ['GET'])]
    public function getParametrage(Request $request, ParametrageBancJourRepository $repo): JsonResponse
    {
        $bancId = $request->query->get('bancId');
        $date = $request->query->get('date');

        if (!$bancId || !$date) {
            return $this->json(['error' => 'Missing bancId or date'], 400);
        }

        $param = $repo->findOneBy([
            'banc' => $bancId,
            'date' => new \DateTime($date)
        ]);

        if (!$param) {
            return $this->json(['error' => 'No parametrage found'], 404);
        }

        return $this->json([
            'longueur' => $param->getLongueur(),
            'largeur' => $param->getLargeurMax(),
        ]);
    }
    


}
