<?php

namespace App\Controller;

use App\Entity\Affaire;
use App\Service\ExcelImportService;
use Doctrine\ORM\EntityManagerInterface;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Routing\Attribute\Route;
use Symfony\Component\Security\Csrf\CsrfToken;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\Security\Csrf\CsrfTokenManagerInterface;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;

final class ExcelController extends AbstractController
{
    private $em;
    public function __construct(EntityManagerInterface $em)
    {
        $this->em = $em;
    }
    
    #[Route('/prod/import-excel', name: 'import_excel', methods: ['GET', 'POST'])]
    public function import(Request $request, ExcelImportService $excelImportService, CsrfTokenManagerInterface $csrfTokenManager): JsonResponse|Response {

        ini_set('memory_limit', '512M'); 
        set_time_limit(300); // Augmente la limite à 5 min.

        if ($request->isMethod('POST')) {

            $submittedToken = $request->request->get('_csrf_token');

            if (!$csrfTokenManager->isTokenValid(new CsrfToken('excel_import', $submittedToken))) {
                return new JsonResponse(['error' => 'Token CSRF invalide.'], 403);
            }

            $file = $request->files->get('excelFile');
            $force = $request->request->getBoolean('force');
            $originalFilename = $file->getClientOriginalName();
            
            if (!$file) {
                return new JsonResponse(['error' => 'Aucun fichier reçu.'], 400);
            }

            if (!in_array($file->guessExtension(), ['xls', 'xlsx', 'xlsm'])) {
                return new JsonResponse(['error' => 'Type de fichier invalide.'], 400);
            }
            // RECUP DATE DISPO
            $dispoDateStr = $request->request->get('dispoDate');
            $dispoDate = null;

            if (empty($dispoDateStr)) {
                return new JsonResponse(['error' => 'La date de mise à disposition est obligatoire.'], 400);
            }

            try {
                $dispoDate = new \DateTime($dispoDateStr);
                $now = new \DateTime('today');

                if ($dispoDate <= $now) {
                    return new JsonResponse(['error' => 'La date de mise à disposition doit être ultérieure à aujourd’hui.'], 400);
                }
            } catch (\Exception $e) {
                return new JsonResponse(['error' => 'Date invalide.'], 400);
            }

            $affaireId = $request->request->get('affaireId');

            if ($affaireId) {
                $affaire = $this->em->getRepository(Affaire::class)->find($affaireId);
                if (!$affaire) {
                    return new JsonResponse(['error' => 'Affaire introuvable.'], 400);
                }
            } else {
                $affaire = null;
            }            


            try {
                $data = $excelImportService->importExcelData($file->getPathname(), $originalFilename, $force, $dispoDate, $affaire);
            } catch (\Throwable $e) {
                return new JsonResponse(['error' => 'Erreur lors du traitement Excel : '.$e->getMessage()], 500);
            }

            return new JsonResponse([
                'success' => true,
                'originalFilename' => $originalFilename,
                'data' => $data,
            ], 200);
        }
        $affaires = $this->em->getRepository(Affaire::class)->getListeAffaires();        

        return $this->render('excel/index.html.twig', [
        'affaires' => $affaires
        ]);

    }
}
