<?php

namespace App\Controller;

use App\Entity\Affaire;
use App\Entity\DetailAffaire;
use App\Entity\Regroupement;
use Doctrine\ORM\EntityManagerInterface;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Routing\Attribute\Route;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\Security\Csrf\CsrfTokenManagerInterface;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;

final class DetailAffaireController extends AbstractController
{
    private $em;
    public function __construct(EntityManagerInterface $em)
    {
        $this->em = $em;
    }
    
    #[Route('/prod/detail/affaire', name: 'app_detail_affaire')]
    public function index(): Response
    {
        return $this->render('detail_affaire/index.html.twig', [
            'controller_name' => 'DetailAffaireController',
        ]);
    }

    #[Route('/prod/detailaffaire/liste/{id}', name: 'app_detail_affaire_liste')]
    public function listeDetailParAffaire(Affaire $affaire): Response
    {
       $detailsAffaire = $this->em->getRepository(DetailAffaire::class)->getListeDetailParAffaire($affaire);
       $detailsAffaireRegroupee = $this->em->getRepository(DetailAffaire::class)->getListeDetailParAffaireRegroupee($affaire);
       
        
        return $this->render('detail_affaire/index.html.twig', [            
            'detailsAffaire' => $detailsAffaire,
            'detailsAffaireRegroupee' => $detailsAffaireRegroupee,
        ]);
    }

    #[Route('/admin/detailaffaire/{id}/delete-token', name: 'detailaffaire_delete_token', methods: ['GET'])]
    public function getDeleteToken(int $id, CsrfTokenManagerInterface $csrfTokenManager): JsonResponse
    {
        $token = $csrfTokenManager->getToken('delete'.$id)->getValue();
        return new JsonResponse(['token' => $token]);
    }

    #[Route('/admin/detailaffaire/{id}/delete', name: 'detailaffaire_delete', methods: ['POST'])]
    public function delete(Request $request, DetailAffaire $detailAffaire): Response
    {
        // Vérification du token CSRF avant de procéder à la suppression
        $submittedToken = $request->request->get('_token');
        if (!$this->isCsrfTokenValid('delete'.$detailAffaire->getId(), $submittedToken)) {
            $this->addFlash('danger', 'Échec de la suppression : token CSRF invalide.');
            return $this->redirectToRoute('app_affaire_modifier', ['id' => $detailAffaire->getAffaire()->getId()]);
        }

        // Suppression de l'entité Banc
        //$entityManager->remove($banc);
        $detailAffaire->setDateDelete(new \DateTime('now'));
        $this->em->persist($detailAffaire);
        $this->em->flush();

        // Message flash de succès
        $this->addFlash('success', "La poutre \"{$detailAffaire->getDesignation()}\" a été supprimée avec succès.");

        // Redirection vers la liste (page index des bancs)
        return $this->redirectToRoute('app_affaire_modifier', ['id' => $detailAffaire->getAffaire()->getId()], Response::HTTP_SEE_OTHER);
    }


   #[Route('/api/detail-affaires', name: 'api_detail_affaires', methods: ['GET'])]
    public function planification(): JsonResponse
    {                
        $detailsAffaire = $this->em->getRepository(DetailAffaire::class)->findPoutresNonPlaceesAvecQuantiteRestante();
        
        // 2. Regroupements
        $regroupements = $this->em->getRepository(Regroupement::class)->findAll();

        // 3. Transformation des poutres individuelles
        $data = array_map(function ($detailsAffaire) {
            return [
                'id' => $detailsAffaire->getId(),
                'designation' => $detailsAffaire->getDesignation(),
                'nom' => $detailsAffaire->getDesignation(),
                'nbPiece' => $detailsAffaire->getNbPiece(),
                'nbLame' => $detailsAffaire->getNbLame(),
                'longueur' => $detailsAffaire->getLongueurProduction(),
                'largeur' => $detailsAffaire->getSectionLameLargeur(),
                'hauteur'=> $detailsAffaire->getSectionLameHauteur(),
                'affaire'=> $detailsAffaire->getAffaire()->getNomChantier(),
                'quantiteRestante' => $detailsAffaire->getQuantiteRestante(),
                'refile' => $detailsAffaire->getGlxxhsRefile(),
                'lamelisation' => $detailsAffaire->getLamelisation(),
                'tac' => $detailsAffaire->getTac(),
                'gl24h' => $detailsAffaire->getGl24h(),
                'gl28h' => $detailsAffaire->getGl28h(),
                'groupes' => [],
            ];
        }, $detailsAffaire);

        // 4. Transformation des regroupements
        foreach ($regroupements as $grp) {
            $details = $grp->getPoutres()->toArray();
            if (count($details) < 2) continue;

            $designation = 'GR-LONG-' . implode('-', array_map(fn($d) => $d->getDesignation(), $details));
            $longueurTotale = array_sum(array_map(fn($d) => $d->getLongueurProduction(), $details));
            $largeurRef = $details[0]->getSectionLameLargeur();

            // On s'assure que toutes les largeurs sont identiques
            if (!array_reduce($details, fn($carry, $d) => $carry && $d->getSectionLameLargeur() === $largeurRef, true)) {
                continue;
            }

            $data[] = [
                'id' => 'GRP-' . $grp->getId(),
                'designation' => $designation,
                'nom' => $designation,
                'nbPiece' => 1,
                'nbLame' => $details[0]->getNbLame(),
                'longueur' => $longueurTotale,
                'largeur' => $largeurRef,
                'hauteur'=> $details[0]->getSectionHauteur(),
                'affaire'=> $grp->getAffaire()?->getNomChantier() ?? '',
                'refile' => false,
                'lamelisation' => true,
                'groupes' => array_map(fn($d) => [
                    'id' => $d->getId(),
                    'designation' => $d->getDesignation(),
                    'longueur' => $d->getLongueurProduction(),
                    'largeur' => $d->getSectionLameLargeur(),
                ], $details),
            ];
        }

        return $this->json($data);
    }

}
