import React, { useEffect, useState, useRef } from "react";
import { DndProvider, useDrag, useDrop } from "react-dnd";
import { HTML5Backend } from "react-dnd-html5-backend";
import { v4 as uuidv4 } from "uuid";
import Accordion from "react-bootstrap/Accordion";
import DatePicker from "react-datepicker";
import "react-datepicker/dist/react-datepicker.css";
import { registerLocale } from "react-datepicker";
import fr from "date-fns/locale/fr"; // 🇫🇷 locale française
registerLocale("fr", fr);


/**
 * Planification React – version complète et stable
 * ------------------------------------------------
 * • Responsive (100% × 60vh)
 * • Filtres, tri, sélection multiple, regroupement, dissociation
 * • Placement déterministe (colonne la plus à gauche, empilement vertical)
 * • Stock automatique
 * • Aucune mutation directe de l'état => stabilité
 */

const ItemTypes = { POUTRE: "poutre" };
const DISPLAY_RATIO = {
  longueur: 0.05, // 1 mm = 0.05 px → 20000 mm = 1000 px
  largeur: 0.2,   // 1 mm = 0.3 px → 1980 mm = 990 px
};

/* -------------------------------------------------------------------------- */
/* UI HELPERS                                                                 */
/* -------------------------------------------------------------------------- */
function Card({ children, className = "" }) {
  return <div className={`border rounded shadow bg-white p-4 mb-4 ${className}`}>{children}</div>;
}

function Button({ children, onClick, className = "", disabled = false }) {
  return (
    <button
      onClick={(e) => {
        e.preventDefault();
        if (!disabled) onClick?.();
      }}
      disabled={disabled}
      className={`px-4 py-2 rounded text-white transition 
        ${disabled ? "bg-gray-400 cursor-not-allowed" : "bg-blue-600 hover:bg-blue-700"} 
        ${className}`}
    >
      {children}
    </button>
  );
}


/* -------------------------------------------------------------------------- */
/* POUTRE – DRAG SOURCE                                                       */
/* -------------------------------------------------------------------------- */
function Poutre({ poutre, onSelect, selected }) {
  const isDisabled = poutre.refile == 1 && (poutre.quantiteRestante ?? 0) < 2;

  const [{ isDragging }, drag] = useDrag(() => ({
    type: ItemTypes.POUTRE,
    item: {
      ...poutre,
      poutreId: poutre.poutreId ?? poutre.id,
    },
    canDrag: !isDisabled,
    collect: (monitor) => ({ isDragging: monitor.isDragging() })
  }));

  const designation = poutre.designation || "(sans désignation)";
  const bgColor = 
    designation.startsWith("STOCK") ? "bg-yellow-100 border-yellow-400" :
    designation.startsWith("CHUTE") ? "bg-blue-100 border-blue-400" :
    poutre.groupes?.length          ? "bg-green-100 border-green-400" :
    poutre.refile == 1              ? "bg-success" :
    poutre.lamelisation == 1        ? "bg-warning" :
                                      "bg-white";
  const largeurEffective = Number(poutre.largeur) * Number(poutre.nbLame || 1);
  return (
    <div
      ref={drag}
      onClick={() => onSelect?.(poutre)}
      className={`p-2 border rounded shadow mb-2 cursor-pointer ${selected ? "bg-blue-100 border-blue-400" : bgColor} ${isDisabled ? "opacity-50 cursor-not-allowed" : ""}`}
      style={{ opacity: isDragging ? 0.5 : 1 }}
    >
      {poutre.quantiteRestante != null && poutre.nbPiece ? `(${poutre.nbPiece - poutre.quantiteRestante}/${poutre.nbPiece})` : ""} : {designation} – {poutre.longueur} × {largeurEffective} x {poutre.hauteur} mm ({poutre.nbLame} lames x {poutre.largeur} ) – {typeof poutre.affaire === "string" ? poutre.affaire : (poutre.affaire?.nom ?? "")} {poutre.lamelisation == 1 ? "– * L *" : ""} {poutre.refile == 1 ? "– * R *" : ""}      
    </div>
  );
}



/* -------------------------------------------------------------------------- */
/* ALGORITHME DE PLACEMENT                                                    */
/* -------------------------------------------------------------------------- */
function computePositions(items, longueurMax, largeurMax, modeObjet = false) {
  const columns = [];
  const positioned = [];
  let erreurPlacement = 0;

  for (const beam of items) {
  const largeurEffective = Number(beam.largeur) * Number(beam.nbLame || 1);
  const longueur = Number(beam.longueur);

  // ✅ Ne repositionne pas une poutre déjà placée manuellement
  if (beam.isStockAuto && beam.left != null && beam.top != null) {
    positioned.push({ ...beam });
    continue;
  }


  let column = columns.find(
    (c) => c.width === largeurEffective && c.height + longueur <= longueurMax
  );

  if (!column) {
    const usedWidth = columns.reduce((s, c) => s + c.width, 0);
    const largeurRestante = largeurMax - usedWidth;

    if (largeurEffective > largeurRestante) {
      erreurPlacement = 10;
      continue;
    }

    if (longueur > longueurMax) {
      erreurPlacement = 11;
      continue;
    }

    column = { width: largeurEffective, height: 0 };
    columns.push(column);
  }

  const left = columns
    .slice(0, columns.indexOf(column))
    .reduce((s, c) => s + c.width, 0);
  const top = column.height;

  const id = beam.id ?? uuidv4();
  positioned.push({ ...beam, id, left, top });
  column.height += longueur;
}

  
  return modeObjet
    ? { positioned, erreur: erreurPlacement }
    : positioned;
}

/* -------------------------------------------------------------------------- */
/* BANC – DROP TARGET                                                         */
/* -------------------------------------------------------------------------- */
function Banc({ couche, poutres, onDrop, onRemove, longueurMax, largeurMax }) {
  const [{ isOver }, drop] = useDrop(() => ({
    accept: ItemTypes.POUTRE,
    drop: (item) => onDrop(item, couche),
    collect: (m) => ({ isOver: m.isOver() })
  }));

  const positioned = computePositions(poutres, longueurMax, largeurMax);

  // Regrouper les poutres par colonne réelle (via .left)
  // Colonnes par .left
const colonnes = {};
for (const p of positioned) {
  const cle = (p.left || 0).toFixed(2);
  if (!colonnes[cle]) {
    colonnes[cle] = [];
  }
  colonnes[cle].push(p);
}

// Longueur = max hauteur des colonnes
const hauteurs = Object.values(colonnes).map(col =>
  col.reduce((s, p) => s + Number(p.longueur), 0)
);
const longueurUtilisee = Math.max(0, ...hauteurs);

// Largeur = somme des largeurs des colonnes uniques
const largeurUtilisee = Object.values(colonnes).reduce((s, col) => {
  const p = col[0];
  return s + (Number(p.largeur) * Number(p.nbLame || 1));
}, 0);


  const styleContainer = {
    width: `${largeurMax * DISPLAY_RATIO.largeur}px`,
    height: `${longueurMax * DISPLAY_RATIO.longueur}px`,
    position: "relative",
    backgroundColor: "#f9fafb",
    border: isOver ? "2px solid #3b82f6" : "1px solid #d1d5db",
    overflow: "hidden"
  };

  return (
    <div className="border p-2 bg-gray-100 w-full">
      <div className="mb-2 font-bold">Couche {couche.toUpperCase()}</div>
      <div className="bg-gray-200 p-2 mb-2 text-sm">
        Long. {longueurUtilisee}/{longueurMax} mm · Larg. {largeurUtilisee}/{largeurMax} mm
      </div>

      <div ref={drop} style={styleContainer}>
        {positioned.map((p) => {
          console.log("Render", p.designation, "→ left:", p.left, "→ leftPx:", Number(p.left) * DISPLAY_RATIO.largeur);

          const largeurEffective = Number(p.largeur) * Number(p.nbLame || 1);
          const leftPx = Number(p.left) * DISPLAY_RATIO.largeur;
          const topPx = Number(p.top) * DISPLAY_RATIO.longueur;
          const widthPx = largeurEffective * DISPLAY_RATIO.largeur;
          const heightPx = Number(p.longueur) * DISPLAY_RATIO.longueur;

          const designation = p.designation || "(sans désignation)";
          const bg = designation.startsWith("CHUTE")
                    ? "#FCD34D" // orange
                    : designation.startsWith("STOCK")
                      ? "#D1FAE5" // vert
                      : p.groupes?.length
                        ? "#BFDBFE"
                        : "#E5E7EB";


          return (
            <div
              key={p.id}
              onDoubleClick={() => onRemove(p)}
              className="absolute text-[10px] leading-tight text-center border border-black overflow-hidden font-mono"
              style={{
                left: `${leftPx}px`,
                top: `${topPx}px`,
                width: `${widthPx}px`,
                height: `${heightPx}px`,
                backgroundColor: bg,
                minWidth: 10,
                minHeight: 10
              }}
            >
              {designation}
              <br />({p.longueur}×{largeurEffective})
              <br />{typeof p.affaire === "string" ? p.affaire : (p.affaire?.nom ?? "")}
              <button
                onClick={() => onRemove(p)}
                className="absolute bottom-0 right-0 px-1 text-red-600"
              >×</button>
            </div>
          );
        })}
      </div>

      <Button className="mt-2" onClick={() => positioned.forEach(onRemove)}>
        Réinitialiser cette couche
      </Button>
    </div>
  );
}


/* -------------------------------------------------------------------------- */
/* COMPOSANT PRINCIPAL                                                        */
/* -------------------------------------------------------------------------- */
export default function Planification() {
  /* -------------------- STATES -------------------- */
  const [poutres, setPoutres] = useState([]);           // File d'attente
  const [bancs, setBancs] = useState({}); // Poutres placées
  const [banc, setBanc] = useState(null);
  const [sortKey, setSortKey] = useState("designation");
  const [filter, setFilter] = useState({ designation: "", longueur: "", affaire: "", largeur: "", hauteur: "", largeurEffective: "", typeBois: "tous" });
  const [selectedPoutres, setSelectedPoutres] = useState([]);
  const [dateCollage, setDateCollage] = useState(() => new Date().toISOString().substring(0, 10));
  const [heureCollage, setHeureCollage] = useState("08:00");
  const [planValide, setPlanValide] = useState(false);
  const [bancId, setBancId] = useState(null);
  const nombreCouches = banc?.nombre_couche || 2;
  const [bancDims, setBancDims] = useState({ longueur: 20000, largeur: 1980 }); // par défaut
  const [datesDisponibles, setDatesDisponibles] = useState([]);
  const [poutresInitiales, setPoutresInitiales] = useState([]);
  const bancsRef = useRef(bancs);
  const [planEnregistre, setPlanEnregistre] = useState(false);


  function peutPlacer(beam, couche) {
    const liste = bancs[couche] || [];

    const largeurEffective = Number(beam.largeur) * Number(beam.nbLame || 1);
    const longueur = Number(beam.longueur);

    const largeurUtilisee = liste.reduce(
      (s, p) => s + Number(p.largeur) * Number(p.nbLame || 1),
      0
    );

    const colonnes = [];
    for (const p of liste) {
      const lEff = Number(p.largeur) * Number(p.nbLame || 1);
      const col = colonnes.find(c => c.width === lEff);
      if (col) col.height += p.longueur;
      else colonnes.push({ width: lEff, height: p.longueur });
    }

    const colExistante = colonnes.find(
      (c) => c.width === largeurEffective && c.height + longueur <= bancDims.longueur
    );

    const largeurTotale = colonnes.reduce((s, c) => s + c.width, 0);

    const peutCreerColonne = largeurTotale + largeurEffective <= bancDims.largeur;

    return colExistante || peutCreerColonne;
  }

  function peutVraimentPlacer(beam, couche) {
    const liste = bancs[couche] || [];

    // Simuler le plan actuel + la poutre à ajouter
    const testListe = [...liste, beam];

    const resultat = computePositions(testListe, bancDims.longueur, bancDims.largeur);

    // Si le résultat contient bien la poutre (elle a une position), c’est OK
    return resultat.some(p => p.id === beam.id && p.left != null && p.top != null);
  }

  /* -------------------- RECUPERATION des paramètres (depuis Twig) -------------------- */
  useEffect(() => {
    const planifData = window.PLANIF;
    if (planifData) {      
      setBancId(planifData.bancId);
      setDateCollage(planifData.date);
      setHeureCollage(planifData.heure || "08:00");
    }
  }, []);
  /* -------------------- RECUPERATION DU BANC -------------------- */
  useEffect(() => {
    if (!bancId) return;

    fetch(`/api/bancs/${bancId}`)
      .then((res) => res.json())
      .then((data) => setBanc(data))
      .catch((err) => console.error("Erreur lors du chargement du banc :", err));
  }, [bancId]);

  useEffect(() => {
    if (!bancId || !dateCollage) return;
    
    fetch(`/api/parametrage-banc-jour?bancId=${bancId}&date=${dateCollage}&heure=${heureCollage}`)

      .then((res) => {
        if (!res.ok) throw new Error("Aucun paramétrage trouvé");
        return res.json();
      })
      .then((data) => {
        const { longueur, largeur } = data;
        if (longueur && largeur) {
          setBancDims({ longueur: Number(longueur), largeur: Number(largeur) });
        }
      })
      .catch((err) => {
        console.warn("Paramétrage du banc non trouvé, on garde les valeurs par défaut", err);
        setBancDims({ longueur: 20000, largeur: 1980 }); // fallback
      });
  }, [bancId, dateCollage]);

  useEffect(() => {
      if (!banc || Object.keys(bancs).length > 0) return;

    // Pas de plan chargé automatiquement => on initialise les couches
    fetch(`/api/plans-collage?bancId=${banc.id}&date=${dateCollage}&heure=${heureCollage}`)
      .then(res => {
        if (!res.ok) {
          // Aucun plan => on initialise les couches vides
          const initialBancs = {};
          for (let i = 1; i <= banc.nombre_couche; i++) {
            initialBancs[`couche${i}`] = [];
          }
          setBancs(initialBancs);
        }
      });
  }, [banc]);
/* -------------------- RECUP DES DATES VALIDÉES POUR CE BANC -------------------- */  
  useEffect(() => {
    if (!bancId) return;

    fetch(`/api/dates-parametrees?bancId=${bancId}`)
      .then((res) => res.json())
      .then((data) => setDatesDisponibles(data))
      .catch((err) => console.error("Erreur chargement dates disponibles :", err));
  }, [bancId]);


/* -------------------- CHARGEMENT AUTOMATIQUE DU PLAN -------------------- */  
  useEffect(() => {
  if (!bancId || !dateCollage || !heureCollage) return;

  fetch(`/api/plans-collage?bancId=${bancId}&date=${dateCollage}&heure=${heureCollage}`)
    .then(res => {
      if (!res.ok) {        
        return null;
      }
      return res.json();
      setPlanEnregistre(true); // ✅ Plan initialement enregistré
    })
    .then(data => {
      if (!data || !data.bancs) return;

      const nouveauxBancs = {};
      for (const [couche, poutres] of Object.entries(data.bancs)) {        
        const poutresNumeriques = poutres.map(p => ({
          ...p,
          couche,
          id: p.id ?? p.poutreId ?? uuidv4(),
          longueur: Number(p.longueur),
          largeur: Number(p.largeur),
          left: p.left,
          top: p.top,
          isStockAuto: p.isStockAuto ?? false,
        }));


        const toutesOntPosition = poutresNumeriques.every(p => p.left != null && p.top != null);

        nouveauxBancs[couche] = toutesOntPosition
          ? poutresNumeriques
          : computePositions(poutresNumeriques, bancDims.longueur, bancDims.largeur);
      }

      setBancs(nouveauxBancs);
    })
    .catch(err => {
      console.error("Erreur de chargement du plan :", err);
    });
  }, [bancId, dateCollage, heureCollage]);

  /* -------------- INITIAL LOAD -------------- */
  useEffect(() => {
    fetch("/api/detail-affaires")
      .then((res) => res.json())
      .then((data) => {
        const liste = data.map((d) => ({
          ...d,
          id: uuidv4(),
          poutreId: d.id,
          groupes: d.groupes || [],
          quantiteRestante: d.nbPiece ?? 1,
          nbPiece: d.nbPiece ?? 1,
          tac: !!d.tac,
          gl24h: !!d.gl24h,
          gl28h: !!d.gl28h
        }));
        setPoutres(liste);
        setPoutresInitiales(liste); // ✅ garder l'originale
      });
   }, []);

  /* -------------- SELECTION -------------- */
  const handleSelect = (beam) => {
    setSelectedPoutres((prev) =>
      prev.some((p) => p.id === beam.id) ? prev.filter((p) => p.id !== beam.id) : [...prev, beam]
    );
  };

  /* -------------- DRAG & DROP -------------- */
  const handleDrop = (beam, couche) => {
    const largeurEffective = Number(beam.largeur) * Number(beam.nbLame || 1);
    const longueur = Number(beam.longueur);
    
    if (longueur > bancDims.longueur || largeurEffective > bancDims.largeur) {
      alert("La poutre dépasse les dimensions maximales du banc !");
      bancsRef.current = { ...bancs };
      return;
    }
    
    const copiePoutre = {
      ...beam,
      id: uuidv4(),
      poutreId: beam.poutreId ?? beam.id,
      couche
    };
        
    const simulation = [...(bancsRef.current[couche] || []), copiePoutre];
    const { positioned, erreur: erreurPlacement } = computePositions(simulation, bancDims.longueur, bancDims.largeur, true);    
    
    if (erreurPlacement>0) {
      alert("Pas assez de place pour placer cette poutre sur la couche ");
      bancsRef.current = { ...bancs };
      return;
    }

    // OK, on ajoute
    setBancs((prev) => {
      const newState = {
        ...prev,
        [couche]: [...(prev[couche] || []), copiePoutre]
      };
      bancsRef.current = newState; // ← MAJ de la ref
      return newState;
    });


    // Mise à jour des quantités restantes
    if (!beam.groupes?.length) {
      const quantiteRetiree = beam.refile == 1 ? 2 : 1;

      setPoutres(prev => {
        const updated = prev.map(p => {
          if (p.poutreId === beam.poutreId) {
            const reste = (p.quantiteRestante ?? 1) - quantiteRetiree;
            return { ...p, quantiteRestante: reste };
          }
          return p;
        });

        return updated.filter(p => (p.quantiteRestante ?? 1) > 0);
      });
    }
    setPlanEnregistre(false);

  };

  const handleRemove = (beam) => {
    setBancs((prev) => {
      const newState = {
        ...prev,
        [beam.couche]: prev[beam.couche].filter((p) => p.id !== beam.id)
      };
      bancsRef.current = newState;
      return newState;
      setPlanEnregistre(false); // ❌ plan modifié
    });

    // Si c'était une sous-poutre d’un regroupement, on ne la remet pas
    if (beam.regroupementId) return;

    // Sinon, on restaure la poutre d’origine
    setPoutres((prev) => {
      return prev.map(p => {
        // On retrouve la poutre d’origine via le poutreId
        const ajout = beam.refile == 1 ? 2 : 1;
        if (p.poutreId === beam.poutreId || p.id === beam.poutreId) {
          return {
            ...p,
            quantiteRestante: (p.quantiteRestante ?? 0) + ajout
          };
        }
        return p;
      });
    });
  };


  /* -------------- FILTERS & SORT -------------- */
  const resetFilters = () => setFilter({ designation: "", longueur: "", affaire: "", largeur: "", hauteur: "", largeurEffective: "", typeBois: "tous" });

  const filtered = poutres.filter((p) => {
    const designation = (p.designation || "").toLowerCase();
    const filtreDesignation = (filter.designation || "").toLowerCase();

    const largeurEffective = Number(p.largeur) * Number(p.nbLame || 1);
    const filtreLargeurEffective = filter.largeurEffective || "";

    const longueur = p.longueur?.toString() || "";
    const largeur = p.largeur?.toString() || "";
    const hauteur = p.hauteur?.toString() || "";
    const nbLame = p.nbLame?.toString() || "";

    const filtreLongueur = filter.longueur || "";
    const filtreLargeur = filter.largeur || "";
    const filtreHauteur = filter.hauteur || "";

    const affaireNom = typeof p.affaire === "string"
      ? p.affaire.toLowerCase()
      : (p.affaire?.nom || "").toLowerCase();


    const filtreAffaire = (filter.affaire || "").toLowerCase();

    return (
      designation.includes(filtreDesignation) &&
      (filtreLongueur === "" || longueur.includes(filtreLongueur)) &&
      (filtreLargeur === "" || largeur.includes(filtreLargeur)) &&
      (filtreHauteur === "" || hauteur.includes(filtreHauteur)) &&
      affaireNom.includes(filtreAffaire) &&
      (filtreLargeurEffective === "" || largeurEffective.toString().includes(filtreLargeurEffective)) &&
      (
        filter.typeBois === "tous" ||
        (filter.typeBois === "tac" && p.tac) ||
        (filter.typeBois === "gl24h" && p.gl24h) ||
        (filter.typeBois === "gl28h" && p.gl28h)
      )
    );
  });

  const sortedFiltered = filtered.sort((a, b) => {
//    if (sortKey === "designation") return a.designation.localeCompare(b.designation);
    if (sortKey === "designation") {
      const da = a.designation || "";
      const db = b.designation || "";
      return da.localeCompare(db);
    }

    if (sortKey === "longueur") return (a.longueur || 0) - (b.longueur || 0);
    if (sortKey === "largeur") return (a.largeur || 0) - (b.largeur || 0);
    if (sortKey === "hauteur") return (a.hauteur || 0) - (b.hauteur || 0);

    return 0;
  });

  /* -------------- GROUP / UNGROUP -------------- */
  const regrouperLongueur = () => {
    if (selectedPoutres.length < 2) return;
    const largeurRef = selectedPoutres[0].largeur;
    if (!selectedPoutres.every((p) => p.largeur === largeurRef)) return alert("Largeurs différentes !");

    const group = {
      id: uuidv4(),
      designation: "GR-LONG-" + selectedPoutres.map((p) => p.designation).join("-"),
      longueur: selectedPoutres.reduce((s, p) => s + p.longueur, 0),
      largeur: largeurRef,
      groupes: selectedPoutres,
      affaire: "",
      nbLame: selectedPoutres[0].nbLame || 0
    };

    setPoutres((prev) => [
      ...prev.filter((p) => !selectedPoutres.includes(p)),
      group
    ]);
    setSelectedPoutres([]);
  };

  const regrouperLargeur = () => {
    if (selectedPoutres.length < 2) return;
    const longueurRef = selectedPoutres[0].longueur;
    if (!selectedPoutres.every((p) => p.longueur === longueurRef)) return alert("Longueurs différentes !");

    const group = {
      id: uuidv4(),
      designation: "GR-LARGE-" + selectedPoutres.map((p) => p.designation).join("-"),
      longueur: longueurRef,
      largeur: selectedPoutres.reduce((s, p) => s + p.largeur, 0),
      groupes: selectedPoutres,
      affaire: "",
      nbLame: selectedPoutres[0].nbLame || 0
    };

    setPoutres((prev) => [
      ...prev.filter((p) => !selectedPoutres.includes(p)),
      group
    ]);
    setSelectedPoutres([]);
  };

  const dissocier = () => {
    const regroups = selectedPoutres.filter((p) => p.groupes?.length);
    const originals = regroups.flatMap((p) => p.groupes);
    setPoutres((prev) => [
      ...prev.filter((p) => !regroups.includes(p)),
      ...originals.filter((p, i, self) => self.findIndex((q) => q.id === p.id) === i)
    ]);
    setSelectedPoutres([]);
  };

/* -------------- STOCK AUTO -------------- */
    /* ----------- LONGUEUR ----------- */
  const proposerStockLongueur = () => {
    const bancsActuels = { ...bancs };
    const newBancs = {};

    // 🧠 Étape 1 : calcul hauteur max GLOBALE (toutes couches confondues)
    const hauteurGlobaleMax = Math.max(
      ...Object.values(bancsActuels).map(liste => {
        const positioned = computePositions(liste, bancDims.longueur, bancDims.largeur);
        const colonnes = {};
        positioned.forEach(p => {
          const cle = Math.round(p.left); // ← plus stable que .toFixed(2)
          if (!colonnes[cle]) colonnes[cle] = [];
          colonnes[cle].push(p);
        });

        return Math.max(
          0,
          ...Object.values(colonnes).map(col =>
            col.reduce((s, p) => s + Number(p.longueur), 0)
          )
        );
      })
    );

    // 🧱 Étape 2 : traiter chaque couche
    Object.entries(bancsActuels).forEach(([couche, liste]) => {
      const positioned = computePositions(liste, bancDims.longueur, bancDims.largeur); // ← important

      const colonnes = {};
      positioned.forEach(p => {
        const cle = Math.round(p.left);
        if (!colonnes[cle]) {
          colonnes[cle] = {
            poutres: [],
            width: Number(p.largeur) * Number(p.nbLame || 1),
            left: p.left
          };
        }

        colonnes[cle].poutres.push(p);
      });

      Object.values(colonnes).forEach(col => {
        col.height = col.poutres.reduce((s, p) => s + Number(p.longueur), 0);
      });

      const hauteurMax = hauteurGlobaleMax;
      let newListe = [...liste];

      Object.values(colonnes).forEach(col => {
        const manque = hauteurMax - col.height;
        if (manque >= 100) {
          const top = col.poutres.reduce((s, p) => s + Number(p.longueur), 0);

          newListe.push({
            id: uuidv4(),
            designation:
              manque >= 3000
                ? `STOCK-${couche}-${Date.now()}`
                : `CHUTE-${couche}-${Date.now()}`,
            longueur: manque,
            largeur: col.width,
            nbLame: 0,
            groupes: [],
            affaire: "",
            couche,
            isStockAuto: true,
            left: col.left,
            top: top
          });
        }
      });

      newBancs[couche] = newListe;
    });

    setBancs(newBancs);
    bancsRef.current = newBancs;
  };


/* ----------- Largeur ----------- */
  const proposerStockLargeur = () => {
    const bancsActuels = { ...bancs };
    const largeursParCouche = {};
    const longueursParCouche = {};
    const newBancs = {};

    Object.entries(bancsActuels).forEach(([couche, liste]) => {
      const positioned = computePositions(liste, bancDims.longueur, bancDims.largeur);
      const largeurTotale = positioned.reduce((s, p) => s + Number(p.largeur) * Number(p.nbLame || 1), 0);
      const colonnes = {};
      for (const p of positioned) {
        const cle = (p.left || 0).toFixed(2);
        if (!colonnes[cle]) colonnes[cle] = [];
        colonnes[cle].push(p);
      }
      const hauteurMax = Math.max(...Object.values(colonnes).map(col => col.reduce((s, p) => s + Number(p.longueur), 0)), 0);

      largeursParCouche[couche] = largeurTotale;
      longueursParCouche[couche] = hauteurMax;
      newBancs[couche] = positioned;
    });

    const largeurMax = Math.max(...Object.values(largeursParCouche));

    Object.entries(newBancs).forEach(([couche, liste]) => {
      const manque = largeurMax - (largeursParCouche[couche] || 0);
      if (manque > 50) {
        const poutre = {
          id: uuidv4(),
          designation: `STOCK-LARGEUR-${couche}-${Date.now()}`,
          longueur: longueursParCouche[couche] || 1000,
          largeur: manque,
          nbLame: 0,
          groupes: [],
          affaire: "",
          couche,
          isStockAuto: true
        };
        newBancs[couche] = computePositions([...liste, poutre], bancDims.longueur, bancDims.largeur);
      }
    });

    setBancs(newBancs);
    bancsRef.current = newBancs;
  };

  /* -------------- SAVE PLAN -------------- */
  const savePlan = async (isCopie = false, isRedirect = false) => {
    const positionner = (liste) => computePositions(liste, bancDims.longueur, bancDims.largeur);
    

    const payload = {
      dateCollage,
      heureCollage,
      bancId: banc.id,
      valide: planValide,
      ratioLongueur: DISPLAY_RATIO.longueur,
      ratioLargeur: DISPLAY_RATIO.largeur,
      longueurBanc : bancDims.longueur,
      largeurBanc : bancDims.largeur,
      copie: isCopie,
      redirectAfter: isRedirect,
      
      bancs: Object.fromEntries(
        Object.entries(bancs).map(([couche, liste]) => [
          couche,
          computePositions(liste, bancDims.longueur, bancDims.largeur).flatMap((p) => {
            if (p.groupes?.length) {
              return p.groupes.map(g => ({
                poutreId: g.id,
                left: p.left,
                top: p.top,
                regroupementId: p.id
              }));
            }
            // 👉 Traitement spécial pour les poutres STOCK AUTO
            if (p.isStockAuto) {
              return [{
                poutreId: null,
                left: p.left,
                top: p.top,
                regroupementId: null,
                isStockAuto: true,
                longueur: p.longueur,
                largeur: p.largeur,
                couche: p.couche,
                designation: p.designation
              }];
            }

            return [{
              poutreId: p.poutreId ?? p.id,
              left: p.left,
              top: p.top,
              regroupementId: null
            }];
          })
        ])
      )

      
    };

    try {
      const res = await fetch("/api/plans-collage", {
        method: "POST",
        headers: { "Content-Type": "application/json" },
        body: JSON.stringify(payload),
      });

      if (res.ok) {
        const result = await res.json();
        setPlanEnregistre(true); 
        if (result.isRedirect) {
          window.location.href = result.redirect;
        } else {
          alert("Plan enregistré !");
        }
      } else {
        const error = await res.json();
        alert(error.error || "Erreur d'enregistrement");
      }
    } catch (e) {
      console.error("Erreur fatale :", e);
      alert("Erreur réseau ou serveur.");
    }

  };

  /* -------------- RENDER -------------- */
 return (
  <DndProvider backend={HTML5Backend}>
    <div className="p-4 grid grid-cols-2 gap-4">
      {/* LISTE DES POUTRES */}
      <Card>
        <h2 className="text-lg font-bold mb-2">Liste des poutres</h2>

        {/* Barre d’actions, filtres, boutons, etc. */}        
        <div className="flex flex-wrap gap-2 mb-2">          
          <input
            className="border p-1 rounded"
            placeholder="Filtrer longueur"
            value={filter.longueur}
            onChange={(e) => setFilter({ ...filter, longueur: e.target.value })}
          />
          <input
            className="border p-1 rounded"
            placeholder="Filtrer largeur lame"
            value={filter.largeur}
            onChange={(e) => setFilter({ ...filter, largeur: e.target.value })}
          />
          <input
            className="border p-1 rounded"
            placeholder="Filtrer hauteur"
            value={filter.hauteur}
            onChange={(e) => setFilter({ ...filter, hauteur: e.target.value })}
          />
          <input
            className="border p-1 rounded"
            placeholder="Filtrer largeur Poutre"
            value={filter.largeurEffective}
            onChange={(e) => setFilter({ ...filter, largeurEffective: e.target.value })}
          />
        </div>
        <div className="flex flex-wrap gap-2 mb-2">
          <input
            className="border p-1 rounded"
            placeholder="Filtrer désignation"
            value={filter.designation}
            onChange={(e) => setFilter({ ...filter, designation: e.target.value })}
          />
          <input
            className="border p-1 rounded"
            placeholder="Filtrer affaire"
            value={filter.affaire}
            onChange={(e) => setFilter({ ...filter, affaire: e.target.value })}
          />
          
          <select
            className="border p-1 rounded"
            value={filter.typeBois}
            onChange={(e) => setFilter({ ...filter, typeBois: e.target.value })}
          >
            <option value="tous">Tous les bois</option>
            <option value="tac">TAC</option>
            <option value="gl24h">GL24h</option>
            <option value="gl28h">GL28h</option>
          </select>

          <select
            className="border p-1 rounded"
            value={sortKey}
            onChange={(e) => setSortKey(e.target.value)}
          >
            <option value="designation">Trier par désignation</option>
            <option value="longueur">Trier par longueur</option>
            <option value="largeur">Trier par largeur</option>
            <option value="hauteur">Trier par hauteur</option>
          </select>
          <Button onClick={resetFilters}>Réinitialiser Filtres</Button>
        </div>
        <div className="flex flex-wrap gap-2 mb-2">
          <input
            type="time"
            className="border p-1 rounded"
            value={heureCollage}
            onChange={(e) => setHeureCollage(e.target.value)}
          />
          <label className="flex items-center space-x-2">
            <input
              type="checkbox"
              checked={planValide}
              onChange={(e) => setPlanValide(e.target.checked)}
            />
            <span>Plan validé</span>
          </label>
                 
        </div>
        <div className="flex flex-wrap gap-2 mb-2">
          {/* <Button onClick={regrouperLongueur}>Regrouper Longueur</Button>          
          <Button onClick={dissocier}>Dissocier</Button> */}
          <Button className="bg-green-600 hover:bg-green-700" onClick={() => savePlan(false)}>Enregistrer plan</Button>
          <i class="fe fe-arrow-right-circle"></i> 
          <Button onClick={proposerStockLongueur} disabled={!planEnregistre} >Combler Longueur</Button>
          <Button onClick={proposerStockLargeur} disabled={!planEnregistre} >Combler Largeur</Button>  
          <i class="fe fe-arrow-right-circle"></i>
          <Button className="bg-green-800 hover:bg-green-900" onClick={() => savePlan(false, true)}>Enregistrer et Quitter</Button>
        </div>
        <div className="flex flex-wrap gap-2 mb-2">
          <DatePicker
            selected={new Date(dateCollage)}
            onChange={(date) => setDateCollage(date.toISOString().substring(0, 10))}
            className="border p-1 rounded"
            includeDates={datesDisponibles.map(d => new Date(d))}
            locale="fr"
            dateFormat="dd/MM/yyyy"              
          />
          <Button className="bg-orange-500 hover:bg-orange-400" onClick={() => savePlan(true)}>Copier ce plan</Button>
          
        </div>

        {sortedFiltered
          .filter((p) =>
            (p.refile == 1 && (p.quantiteRestante ?? 0) >= 2) || // refilées : min 2 pièces
            (p.refile != 1 && (p.quantiteRestante ?? 0) >= 1)    // autres : min 1 pièce
          )
          .map((p) => (
            <Poutre
              key={p.id}
              poutre={p}
              onSelect={handleSelect}
              selected={selectedPoutres.includes(p)}
            />
        ))}

      </Card>

      {/* COUCHES DYNAMIQUES */}
      <Card>
        <h2 className="text-lg font-bold mb-2">Plan de collage</h2>
        <div className="flex gap-2 mb-4">
          <label>
            Longueur du banc :
            <input
              type="number"
              value={bancDims.longueur}
              onChange={(e) => setBancDims({ ...bancDims, longueur: Number(e.target.value) })}
              className="border p-1 rounded ml-2"
            />
          </label>
          <label>
            Largeur du banc :
            <input
              type="number"
              value={bancDims.largeur}
              onChange={(e) => setBancDims({ ...bancDims, largeur: Number(e.target.value) })}
              className="border p-1 rounded ml-2"
            />
          </label>
        </div>

        <div className="flex flex-wrap gap-4">
          {Array.from({ length: nombreCouches }, (_, i) => {
            const coucheKey = `couche${i + 1}`;
            return (
              <div key={coucheKey} className="flex-1 min-w-[400px]">
                <h3 className="text-md font-semibold mb-2">Couche {i + 1}</h3>
                <Banc
                  couche={coucheKey}
                  poutres={bancs[coucheKey] || []}
                  onDrop={handleDrop}
                  onRemove={handleRemove}
                  longueurMax={bancDims.longueur}
                  largeurMax={bancDims.largeur}
                />
              </div>
            );
          })}
        </div>


      </Card>
    </div>
  </DndProvider>
);

}